
 /*global define, nmlImpl */
/*jslint white: true */

/*
	This class provides the an event handler to zoom and scroll a view.

	Graphs:
		"Mouse/Path/Distance"
		"Mouse/Path"
		"Mouse/Position"	(virtual, for accessing the value and based on Mouse/Path/Distance and Mouse/Path)
*/


define(["src/math/curveUtils", "src/events/keyCodes", "src/utils", "src/build/Builder"],
function (curveUtils, keyCodes, utils, Builder) {
	"use strict";

	// May return null.
	function getVirtualGraphKey (inId) {
		var inIdParts = inId.split("/Position");
		if (inIdParts.length === 2 && inIdParts[1].length === 0) {
			return inIdParts[0];
		}
		return null;
	}

	function getVirtualGraphValue (inId, inT, inX0, inEvalObj) {
		var virtualKey0 = getVirtualGraphKey(inId), result;

		if (virtualKey0) {
			result = getVirtualGraphValue(virtualKey0 + "/Path/Distance", inT, inX0, inEvalObj);
			if (result !== undefined) {
				return getVirtualGraphValue(virtualKey0 + "/Path", inT, result, inEvalObj);
			}
		} else {
			return inEvalObj.getGraphValue(inId, inT, inX0);
		}
	}

	function EventEvaluator (inEventGraphs, inBehaviorInstanceId0, inTimeSpan0, inEventGraphPileItem0) {
		this.eventGraphs = inEventGraphs;
		this.behaviorInstanceId0 = inBehaviorInstanceId0;
		this.timeSpan0 = inTimeSpan0;
		this.eventGraphPileItem0 = inEventGraphPileItem0;
		this.blendWeight = 1.0;

		if (!this.evaluatesAtGlobalTime()) {
			// Initialize the weight based on the event graph pile item.
			var t = this.timeSpan0.t;
			this.blendWeight = this.eventGraphPileItem0.getBlendWeight(t);
		}
	}

	// methods on the param block
	utils.mixin(EventEvaluator, {

		validateEvaluationTime : function (inTime) {
			var	t = inTime, minSampleTime0 = this.eventGraphPileItem0 && this.eventGraphPileItem0.getMinSampleTime(),
				maxSampleTime0 = this.eventGraphPileItem0 && this.eventGraphPileItem0.getMaxSampleTime();

			utils.assert(this.evaluatesAtGlobalTime() || this.timeSpan0, "undefined evaluator time span");

			if (minSampleTime0 !== undefined && t < minSampleTime0) {
				t = minSampleTime0;
			}
			if (maxSampleTime0 !== undefined && t > maxSampleTime0) {
				t = maxSampleTime0;
			}
			return t;
		},
		
		evaluatesAtGlobalTime : function () {
			return !this.behaviorInstanceId0 || !(this.eventGraphPileItem0 && this.eventGraphPileItem0.getFilePath()) ||
					!this.timeSpan0;
		},
		
		getTimeSpan : function (inTimeStack) {
			if (this.evaluatesAtGlobalTime()) {
				return inTimeStack.getRootTimeSpan();
			} else if (this.timeSpan0) {
				return this.timeSpan0;
			}

			utils.assert(this.timeSpan0, "undefined evaluator time span");

			return null;
		},

		// Handles only non-virtual graphs, doesn't conform id or validate time.
		getRawEventGraphValue : function (inGraphId, inTime, inX0) {
		    var filePath0 = this.eventGraphPileItem0 && this.eventGraphPileItem0.getFilePath(), 
                    		    paramAndTakeGroupId0 = this.eventGraphPileItem0 && this.eventGraphPileItem0.getParamAndTakeGroupId(),
								editedKeyframeBucket;
			if (this.behaviorInstanceId0 && filePath0) {
				var editedKeyframeBuckets0 = this.eventGraphPileItem0.getEditedKeyframeBuckets();
				
				if (editedKeyframeBuckets0 && editedKeyframeBuckets0.getEventGraphKeyArray().indexOf(inGraphId) >= 0) {
				    var bucketIndex = editedKeyframeBuckets0.getBucketIndexFromTime(inTime), bucket0,
				        lastBucketIndex = editedKeyframeBuckets0.getLastBucketIndex();
					
					if (lastBucketIndex < bucketIndex) {
						var lastBucketTime = editedKeyframeBuckets0.getTimeFromBucketIndex(lastBucketIndex+1);
						inTime = lastBucketTime;
						bucketIndex = lastBucketIndex;
						// Anything after the last bucket should be pinned to the end of the last bucket.
					}
					
					bucket0 = editedKeyframeBuckets0.getBucket(bucketIndex);
					if (bucket0) {
						var v = editedKeyframeBuckets0.getContinuationValueIfAvailable(bucket0, inTime);
						if (v !== undefined) {
							return v;
						}
						if (bucket0.filePath) {
							filePath0 = bucket0.filePath;
							inTime -= editedKeyframeBuckets0.getTimeFromBucketIndex(bucketIndex);	// time is relative to the start time.
						}
					} else if (inTime > editedKeyframeBuckets0.getSrcFileDuration()) {
						// For editable parameters, if there's no bucket and we are accessing time past the original
						// arf file, pin time to the end of the original file.  -jacquave 9/8/2017
						inTime = editedKeyframeBuckets0.getSrcFileDuration();
					}
				}
			    return nmlImpl.getEventGraphValue(this.eventGraphs.getId(), this.behaviorInstanceId0, filePath0, inGraphId, paramAndTakeGroupId0, inTime, inX0);
			} else {
				// This is a global eval, a direct call here to the graph saves time.
				return this.eventGraphs.getGraphValue(inGraphId, undefined, inX0);
			}
		},

		// Conform id to override pileItem's behaviorInstance with that of the behavior.
		getConformedGraphId : function (inGraphId) {
			if (this.eventGraphPileItem0 && this.eventGraphPileItem0.getBoundInstanceId() &&
					this.behaviorInstanceId0 !== this.eventGraphPileItem0.getBoundInstanceId()) {
				var behaviorInstanceIdStr = "/" + this.behaviorInstanceId0;
				var pileItemBehaviorInstanceIdStr = "/" + this.eventGraphPileItem0.getBoundInstanceId();
				var newId = inGraphId.replace(behaviorInstanceIdStr, pileItemBehaviorInstanceIdStr);
				return newId;
			}
			return inGraphId;
		},

		// Handles only non-virtual graphs
		getGraphValue : function (inGraphId, inTime, inX0) {
			var t = this.validateEvaluationTime(inTime), localId = this.getConformedGraphId(inGraphId);
			return this.getRawEventGraphValue(localId, t, inX0);
		},

		getGraphValues : function (inParentGraphId, inGraphIdsA, inTime, inX0) {
			var t = this.validateEvaluationTime(inTime), localParentId = this.getConformedGraphId(inParentGraphId), 
				resultsA = [], that = this;
			inGraphIdsA.forEach(function (graphId) {
				resultsA.push(that.getRawEventGraphValue(localParentId + graphId, t, inX0));
			});
			return resultsA;
		},

		getGraphMatrix : function (inParentGraphId, inTime) {
			var handleMtx0, mtxData = [], valuesA;

			valuesA = this.getGraphValues(inParentGraphId, ["M0","M1","M3","M4","M6","M7"], inTime);

			// Publish a 2x3 matrix
			mtxData[0] = valuesA[0];
			mtxData[1] = valuesA[1];
			mtxData[2] = 0;
			mtxData[3] = valuesA[2];
			mtxData[4] = valuesA[3];
			mtxData[5] = 0;
			mtxData[6] = valuesA[4];
			mtxData[7] = valuesA[5];
			mtxData[8] = 1;

			if (mtxData[0] !== undefined && mtxData[1] !== undefined && 
				mtxData[3] !== undefined && mtxData[4] !== undefined &&
				mtxData[6] !== undefined && mtxData[7] !== undefined) {
				handleMtx0 = mtxData;
			}

			return handleMtx0;
		},

		// Handles virtual and non-virtual graphs
		getValue : function (inGraphId, inTime) {
			var t = this.validateEvaluationTime(inTime), localId = this.getConformedGraphId(inGraphId);
			return getVirtualGraphValue(localId, t, undefined, this);
		},

		getBlendWeight : function () {
			return this.blendWeight;
		},
	});
	
	return function EventGraphs(inId, inStage0) {
		var that = this, getTimeFunc = nmlImpl.getCurrentTime, strEndsWith, builder = new Builder();

		that.id = inId;

		that.getId = function () {
			return this.id;	
		};
		
		that.resetGraphs = function () {
			this.graphs = {};
			this.incrementalGraphs = {};
			this.lastPathUpdateTime = undefined;
		};

		that.resetGraphs();

		strEndsWith = function (str, suffix) {
			return str.indexOf(suffix, str.length - suffix.length) !== -1;
		};

		that.getEvaluator = function (inBehaviorInstanceId0, inTimeSpan0, inEventGraphPileItem0) {
			return new EventEvaluator(this, inBehaviorInstanceId0, inTimeSpan0, inEventGraphPileItem0);
		};

		that.getNonPathDistanceGraphs = function () {
			var g, pathKey = "Path", pathDistanceKey = "Path/Distance", distGraphKey, result = {};
			for (g in this.graphs) {
				if (this.graphs.hasOwnProperty(g)) {
					if (strEndsWith(g, pathKey)) {
						distGraphKey = g + "/Distance";
						result[g] = { pathGraph: true, graph: this.graphs[g], distGraphKey: distGraphKey, distGraph: this.graphs[distGraphKey] };
					} else if (!strEndsWith(g, pathDistanceKey)) {
						result[g] = { pathGraph: false, graph: this.graphs[g] };
					}
				}
			}

			return result;
		};

		that.deleteOld = function (inOldTime) {
			var nonPathDistGraphs = this.getNonPathDistanceGraphs(), g, graphData, dist;
			for (g in nonPathDistGraphs) {
				if (nonPathDistGraphs.hasOwnProperty(g)) {
					graphData = nonPathDistGraphs[g];

					if (graphData.pathGraph) {
						if (graphData.distGraph) {
							graphData.distGraph.deleteOld(inOldTime);
							dist = this.getGraphValue(graphData.distGraphKey, inOldTime);
							if (dist !== undefined) {
								graphData.graph.deleteOld(dist);
							}
						}
					} else {
						graphData.graph.deleteOld(inOldTime);
					}
				}
			}
		};

		that.update = function (inTime, inOldTime0) {
			nmlImpl.updateEventGraph(that.id, inTime, inOldTime0);
		};

		that.getGraphLastTime = function (inId, inT) {
			var g = this.graphs[inId], result = inT;
			if (g !== undefined) {
				result = g.getXMax();
				if (result === undefined) {
					result = inT;
				}
			}
			return result;
		};

		that.getGraphLastValue = function (inId, inT) {
			var g = this.graphs[inId], result, lastTime = this.getGraphLastTime(inId, inT);
			if (g !== undefined) {
				result = g.evaluate(lastTime);
			}
			return result;
		};

		that.getGraphKeys = function (inKeyPrefix0) {
			// Returns an array of graphs
			var g, grphs = [], matchB;
			for (g in this.graphs) {
				if (this.graphs.hasOwnProperty(g)) {
					matchB = (inKeyPrefix0 === undefined) || g.indexOf(inKeyPrefix0) === 0;
					if (matchB) {
						grphs.push(g);
					}
				}
			}
			return grphs;
		};

		that.getGraph = function (inId) {
			return this.graphs[inId];
		};

		that.hasNonVirtualGraph = function (inId) {
			return this.graphs.hasOwnProperty(inId);
		};

		that.hasGraph = function (inId) {
			var virtualKey0 = getVirtualGraphKey(inId), result;

			if (virtualKey0) {
				result = this.hasNonVirtualGraph(virtualKey0 + "/Path") && this.hasNonVirtualGraph(virtualKey0 + "/Path/Distance");
			} else {
				result = this.hasNonVirtualGraph(inId);
			}
			return result;
		};

		that.getValueExtent = function (inId, inMinX, inMaxX) {
			var result, graph;
			graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getValueExtent(inMinX, inMaxX);
			}

			return result;
		};

		that.getGraphValue = function (inId, inT, inX0) {
			var result, graph, graphXMax;
			graph = this.getGraph(inId);

			if (graph !== undefined) {
				if (inT === undefined) {
					inT = getTimeFunc();
				}
				if (inX0 === undefined) {
					inX0 = inT;
				}
				if (typeof graph.getXMax === "function") {
					graphXMax = graph.getXMax();
					if (inX0 > graphXMax) {
						inX0 = graphXMax; // Get the latest value available.
					}
				}
				result = graph.evaluate(inX0);
			}

			return result;
		};

		that.getValue = function (inGraphId, inTime, inX0) {
			return getVirtualGraphValue(inGraphId, inTime, inX0, this);
		};

		that.getMaxValue = function (inId) {
			var result, graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getYMax();
			}
			return result;
		};

		that.getMinValue = function (inId) {
			var result, graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getYMin();
			}
			return result;
		};

		that.createTimeGraph = function (inGraphId) {
			if (this.graphs[inGraphId] === undefined) {
				this.graphs[inGraphId] = builder.newTimeGraph(undefined);
			}
		};

		that.sustainGraphValue = function (inGraphId, inTime) {
			this.createTimeGraph(inGraphId);
			if (this.hasGraph(inGraphId)) {
				var graph = this.getGraph(inGraphId);
				graph.sustainValue(inTime);
			}
		};

		function appendCurveToGraph(inGraph, inCurve) {
			inGraph.appendCurve(inCurve);
		}

		function appendLineToGraph(inGraph, inLine) {
			inGraph.appendLine(inLine);
		}

		that.setNewHoldValue = function (inId, inTime, inNewValue) {
			var lastValue, lastTime, holdTime, g, segment, oneNanoSecond = 0.000001;
			holdTime = this.getGraphLastTime(inId, undefined);
			if (holdTime !== undefined) {
				holdTime = Math.max(holdTime, inTime - oneNanoSecond);
			}
			this.sustainGraphValue(inId, holdTime);
			lastValue = this.getGraphLastValue(inId, inTime) || inNewValue;
			if (lastValue !== undefined) {
				lastTime = this.getGraphLastTime(inId, inTime);
				g = this.getGraph(inId);
				segment = [[lastTime, lastValue], [inTime, inNewValue]];
				appendLineToGraph(g, segment);
			}
		};

		that.incrementNewHoldValue = function (inId, inTime) {
			var inNewValue = 1, g;
			g = this.getGraph(inId);
			if (g) {
				inNewValue = g.getYMax() + 1;
			}
			this.setNewHoldValue(inId, inTime, inNewValue);
		};

		that.createSpatialTimeGraph = function (inGraphId) {
			if (this.graphs[inGraphId] === undefined) {
				this.graphs[inGraphId] = builder.newSpatialTimeGraph(undefined);
			}
		};

		that.createIncrementalGraph = function (inGraphId, inPos, inSpatialGraphB) {
			var lastCurve;
			if (this.incrementalGraphs[inGraphId] === undefined) {
				this.incrementalGraphs[inGraphId] = new curveUtils.IncrementalLineToCurveCreator(!inSpatialGraphB);
			}
			lastCurve = this.incrementalGraphs[inGraphId].nextPoint(inPos);

			if (lastCurve === undefined) {
				if (inSpatialGraphB) {
					this.createSpatialTimeGraph(inGraphId);
				} else {
					this.createTimeGraph(inGraphId);
				}
			}
		};

		that.updateIncrementalGraph = function (inGraphId, inPos, inSpatialGraphB) {
			var lastCurve;
			this.createIncrementalGraph(inGraphId, inPos, inSpatialGraphB);
			lastCurve = this.incrementalGraphs[inGraphId].getPreviousCurve();

			if (lastCurve !== undefined) {
				appendCurveToGraph(this.getGraph(inGraphId), lastCurve);
			}
		};

		that.removeGraph = function (inGraphId) {
			if (this.graphs[inGraphId] !== undefined) {
				delete this.graphs[inGraphId];
			}
			if (this.incrementalGraphs[inGraphId] !== undefined) {
				delete this.incrementalGraphs[inGraphId];
			}
		};
		
		that.removeGraphs = function (inGraphId) {
			var graphKeysA = this.getGraphKeys(inGraphId);

			if (graphKeysA) {
				var i, k;
				for (i = 0; i < graphKeysA.length; i += 1) {
					k = graphKeysA[i];
					this.removeGraph(k);
				}
			}
		};
		
		that.publishSpatial2D = function (inGraphIdPrefix, inTime, inPos) {
			var d = 0, prevDist, lastPathUpdateTime, distSegment, distGraphId, pathGraphId, pathCurve;

			distGraphId = inGraphIdPrefix + "/Path/Distance";
			pathGraphId = inGraphIdPrefix + "/Path";

			this.createTimeGraph(distGraphId);

			this.createIncrementalGraph(pathGraphId, inPos, true);
			pathCurve = this.incrementalGraphs[pathGraphId].getPreviousCurve();

			if (pathCurve === undefined) {
				// Insert the first point in order to be able to evaluate it.
				pathCurve = curveUtils.create2DCurveFromPoint(inPos);
			}

			prevDist = this.getGraph(pathGraphId).getTotalDistance();
			appendCurveToGraph(this.getGraph(pathGraphId), pathCurve);
			d = this.getGraph(pathGraphId).getTotalDistance();
			
			lastPathUpdateTime = this.getGraphLastTime(distGraphId, inTime);
			distSegment = [[lastPathUpdateTime, prevDist], [inTime, d]];
			appendLineToGraph(this.getGraph(distGraphId), distSegment);
		};

		// for non-spatial
		that.publish1D = function (inGraphId, inTime, inValue, inSustainOldValue0, inIncrementWithMaxValueB) {
			var valueSegment, lastTime, lastValue;

			this.createTimeGraph(inGraphId);

			lastTime = this.getGraphLastTime(inGraphId, inTime);
			lastValue = this.getGraphLastValue(inGraphId, inTime);
			if (lastValue === undefined) {
				lastValue = inValue;
			} else if (inIncrementWithMaxValueB) {
			    inValue = inValue + this.getMaxValue(inGraphId) || 0;
			}

			if (typeof inSustainOldValue0 === "number") {
				if (lastTime) {
					inSustainOldValue0 = (inTime - lastTime) > inSustainOldValue0;
				} else {
					inSustainOldValue0 = false;
				}
			}

			if (inSustainOldValue0) {
				valueSegment = [[lastTime, lastValue], [inTime, lastValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
				valueSegment = [[inTime, lastValue], [inTime, inValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
			} else {
				valueSegment = [[lastTime, lastValue], [inTime, inValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
			}
		};

		that.publishMat3 = function (inGraphParentId, inTime, inMatrix, inSustainOldValueB0) {
			// Publish a 2x3 matrix
		    this.publish1D(inGraphParentId + "M0", inTime, inMatrix[0], inSustainOldValueB0);
			this.publish1D(inGraphParentId + "M1", inTime, inMatrix[1], inSustainOldValueB0);
			this.publish1D(inGraphParentId + "M3", inTime, inMatrix[3], inSustainOldValueB0);
			this.publish1D(inGraphParentId + "M4", inTime, inMatrix[4], inSustainOldValueB0);
			this.publish1D(inGraphParentId + "M6", inTime, inMatrix[6], inSustainOldValueB0);
			this.publish1D(inGraphParentId + "M7", inTime, inMatrix[7], inSustainOldValueB0);
		};	

		that.publishGraphs = function (inGraphDataArray) {
			var i, g;
			for (i = 0; i < inGraphDataArray.length; i += 1) {
				g = inGraphDataArray[i];
				this[g.updateFuncName](g.graphKey, g.time, g.value, g.sustainOldValue, g.incrementWithMaxValue);
			}
		};
		
		that.unpublishGraphs = function (inGraphDataKeysArray) {
			var i;
			for (i = 0; i < inGraphDataKeysArray.length; i += 1) {
				this.removeGraphs(inGraphDataKeysArray[i]);
			}
		};

		that.updateMouseWheelGraph = function (inGraphId, inTime, inDeltaValue) {
			var lastValue;
			that.sustainGraphValue(inGraphId, inTime);
			lastValue = that.getGraphLastValue(inGraphId, inTime) || 0;
			that.setNewHoldValue(inGraphId, inTime, lastValue + inDeltaValue);
		};

		that.isKeyPressed = function (inKeyCodes, inT) {
			var result = false, i;

			for (i = 0; inKeyCodes && !result && i < inKeyCodes.length; i += 1) {
				result = (this.getValue(keyCodes.getKeyGraphId(inKeyCodes[i]), inT) || 0) > 0;
			}

			return result;
		};

		function mouseEventFunc(inEventGraphs) {
			var that = {};

			that.handleEvent = function (event) {
				var pos = event.position.slice(0), t = event.time;
				if (event.type === "out") {
					inEventGraphs.setNewHoldValue("Mouse/Over", t, 0);
				} else {
					if (event.type === "down") {
						if (event.button === "left") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Left", t, 1);
						} else if (event.button === "right") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Right", t, 1);
						} else if (event.button === "middle") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Middle", t, 1);
						}
					} else if (event.type === "up") {
						if (event.button === "left") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Left", t, 0);
						} else if (event.button === "right") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Right", t, 0);
						} else if (event.button === "middle") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Middle", t, 0);
						}
					} else if (event.type === "over") {
						inEventGraphs.sustainGraphValue("Mouse/Path/Distance", t);
						inEventGraphs.setNewHoldValue("Mouse/Over", t, 1);
					} else if (event.type === "wheel" && event.wheelDelta !== undefined) {
						inEventGraphs.updateMouseWheelGraph("Mouse/Wheel/X", t, event.wheelDelta[0]);
						inEventGraphs.updateMouseWheelGraph("Mouse/Wheel/Y", t, event.wheelDelta[1]);
					}

					inEventGraphs.publishSpatial2D("Mouse", t, pos);
				}
				return false;
			};

			return that;
		}

		function touchEventFunc(inEventGraphs) {
			var that = {};

			that.handleEvent = function (event) {
				var t = getTimeFunc(), tKey, touchEvent, idx = 0, idKey, tDown;
				
				for (tKey in event.touches) {
					if (event.touches.hasOwnProperty(tKey)) {
						touchEvent = event.touches[tKey];
						inEventGraphs.setNewHoldValue("Touch/IndexToID/" + idx, t, touchEvent.touchID);
						idx = idx + 1;
						
						idKey = "Touch/ID/" + touchEvent.touchID;
						inEventGraphs.setNewHoldValue(idKey, t, touchEvent.touchID);
						tDown = (touchEvent.type === "down");
						if (tDown || touchEvent.type === "up") {
							inEventGraphs.setNewHoldValue(idKey + "/Down", t, tDown ? 1 : 0);
						}
						
						inEventGraphs.publishSpatial2D(idKey, t, touchEvent.position.slice(0));
					}
				}

				inEventGraphs.setNewHoldValue("Touch/IndexToID/Count", t, idx);
				
				return false;
			};

			return that;
		}

		function updateModKey(inEventKeyValue, inKeyCode, inT, ioEventGraphs) {
			if (inEventKeyValue) {
				ioEventGraphs.incrementNewHoldValue(keyCodes.getKeyGraphId(inKeyCode), inT);
			} else {
				ioEventGraphs.setNewHoldValue(keyCodes.getKeyGraphId(inKeyCode), inT, 0);
			}
		}

		function updateModKeys(inEvent, inT, ioEventGraphs) {
			updateModKey(inEvent.shiftKey, keyCodes.shiftKey, inT, ioEventGraphs);
			updateModKey(inEvent.ctrlKey, keyCodes.cmdKey, inT, ioEventGraphs);
			updateModKey(inEvent.altKey, keyCodes.altKey, inT, ioEventGraphs);
		}

		function keyEventFunc(inEventGraphs) {
			var that = {};

			that.handleEvent = function (event) {
				var handledB = false, t = event.time, graphId, keyGraphs, g;

				if (event.type === "keydown" || event.type === "keyup") {
					var repeatB = event.repeatB;
					graphId = keyCodes.getKeyGraphId(event.keyCode);
					if (event.type === "keydown") {
						if (!repeatB) {
							inEventGraphs.incrementNewHoldValue(graphId, t);
						}
					} else {
						inEventGraphs.setNewHoldValue(graphId, t, 0);
						repeatB = false; // Shouldn't be set, but better safe than sorry.  -jacquave
					}
					if (repeatB) {
						inEventGraphs.incrementNewHoldValue(graphId + "\Repeat", t, 1);
					} else {
						inEventGraphs.setNewHoldValue(graphId + "\Repeat", t, 0);
					}
				} else if (event.type === "gain focus") {
					// Clear all keyboard based event graphs to zero.
					keyGraphs = inEventGraphs.getGraphKeys(keyCodes.getKeyGraphPrefix());

					for (g in keyGraphs) {
						if (keyGraphs.hasOwnProperty(g)) {
							inEventGraphs.setNewHoldValue(keyGraphs[g], t, 0);
						}
					}
					updateModKeys(event, t, inEventGraphs);
				} else if (event.type === "update mod keys") {
					updateModKeys(event, t, inEventGraphs);
				}
				
				// Perhaps we should update mod-keys when we gain focus?  Probably not needed...
				return handledB;
			};

			return that;
		}

		that.timer = function (inTime) {
			this.sustainGraphValue("Mouse/Path/Distance", inTime);
		};

		that.clone = function () {
			var newEventGraph = utils.clone(true, {}, this);
			return newEventGraph;
		};

		if (inStage0) {
			inStage0.installKeyEventHandler(keyEventFunc(this).handleEvent);
			inStage0.installMouseEventHandler(mouseEventFunc(this).handleEvent);
			inStage0.installTouchEventHandler(touchEventFunc(this).handleEvent);
		}

		return that;
	};
});
